--liquibase formatted sql
--changeset ales.holec:231010T1000 stripComments:false splitStatements:false

-- cSpell: ignore translog subq trunc tarnslog plpgsql peanlty setval nextval regclass

-- Fix typos from previous versions
ALTER TABLE IF EXISTS control.zone__perm_ext RENAME pex_peanlty_value TO pex_penalty_value;
ALTER TABLE IF EXISTS control.zone__perm_ext RENAME pex_peanlty_time TO pex_penalty_time;

-- ===================
-- == Payment table ==
-- ===================
DO $$BEGIN raise notice 'Deleting duplicated payments ...'; END;$$;
-- Delete duplicated payments (same pdm, time and tracer)
DELETE FROM control.translog a USING control.translog b
WHERE a.tl_id < b.tl_id  
AND a.tl_pdm_id = b.tl_pdm_id AND a.tl_pay_date_time = b.tl_pay_date_time AND a.tl_tracer=b.tl_tracer;


DO $$BEGIN raise notice 'Adjusting timestamps of non-unique payments ...'; END;$$;
-- Enforce unique pdm_id and payment_timestamp.
-- It is done by adjusting time stamps to which more then one payment for one PDM is assigned.
-- Adjustment is done by sorting all payments within this minute by tracer and using the order (rank) as seconds.
DO
$$
DECLARE
	pay_group record;
BEGIN
	-- Loop over payment id groups having same PDM and timestamp.
	FOR pay_group IN
		-- Find payments groups where multiple payment was done within same time
		WITH pay_group AS (
			SELECT tl_pdm_id as pdm_id, tl_pay_date_time as pay_ts, count (*) as duplication_count
			FROM control.translog
			group by tl_pdm_id, tl_pay_date_time
			HAVING count (*) > 1
		)
		-- We need to adjust all payments within tis minute. Re-group by minute start of minute
		SELECT pdm_id, date_trunc('minute', pay_ts) AS pay_ts_from
		FROM pay_group
		GROUP BY pdm_id, date_trunc('minute', pay_ts)
	LOOP
		-- Update each group by setting seconds, based on order (by tracer) to archive unique time stamp for each payment
		-- Update from select: https://stackoverflow.com/questions/6256610/updating-table-rows-in-postgres-using-subquery
		UPDATE control.translog
		SET tl_pay_date_time = subq.new_ts
		FROM(
			-- Prepare modified timestamps - base minute + number of seconds from order.
			SELECT tl_id as pay_id, date_trunc('minute', tl_pay_date_time) + (INTERVAL '1 second' * RANK() OVER w) as new_ts 
			FROM control.translog
			WHERE tl_pdm_id = pay_group.pdm_id 
			AND tl_pay_date_time >= pay_group.pay_ts_from
			AND tl_pay_date_time < pay_group.pay_ts_from + INTERVAL '1 minute'
			WINDOW w AS (PARTITION BY tl_pdm_id ORDER BY tl_tracer)
		) AS subq
		WHERE tl_id = subq.pay_id;
	END LOOP;
END;
$$;

-- Rename table and fields
DO $$BEGIN raise notice 'Renaming table fields ...'; END;$$;
ALTER TABLE control.translog RENAME TO payment;
ALTER TABLE control.payment RENAME tl_id TO pay_id;
ALTER TABLE control.payment RENAME tl_tracer TO tracer;
ALTER TABLE control.payment RENAME tl_date_pc TO server_ts;
ALTER TABLE control.payment RENAME tl_pay_reason TO pay_reason;
ALTER TABLE control.payment RENAME tl_pay_type TO pay_type;
ALTER TABLE control.payment RENAME tl_pay_date_time TO pay_ts;
ALTER TABLE control.payment RENAME tl_amount TO amount;
ALTER TABLE control.payment RENAME tl_exp_date_time TO expiration_ts;
ALTER TABLE control.payment RENAME tl_ticket_no TO ticket_nr;
ALTER TABLE control.payment RENAME tl_card_nr TO card_nr;
ALTER TABLE control.payment RENAME tl_currency TO currency;
ALTER TABLE control.payment RENAME tl_license_no TO lpn;
ALTER TABLE control.payment RENAME tl_card_code TO card_code;
ALTER TABLE control.payment RENAME tl_parking_space_no TO psn;
ALTER TABLE control.payment RENAME tl_pdm_id TO pdm_id;
ALTER TABLE control.payment RENAME tl_card_type TO card_type;
ALTER TABLE control.payment RENAME tl_authorization_code TO auth_code;
ALTER TABLE control.payment RENAME tl_tariff_info_id TO tariff_info_id;
ALTER TABLE control.payment RENAME tl_rtp_session TO rtp_session;
ALTER TABLE control.payment RENAME tl_custom_data TO custom_data;

-- Rename indexes
ALTER INDEX IF EXISTS control.fki_translog__pdm RENAME TO fki_payment__pdm;
-- Rename constraints
ALTER TABLE IF EXISTS control.payment RENAME CONSTRAINT fk_translog__pdm TO fk_payment__pdm;
ALTER TABLE IF EXISTS control.payment RENAME CONSTRAINT fk_translog__tariff_info TO fk_payment__tariff_info;
ALTER TABLE IF EXISTS control.payment RENAME CONSTRAINT fk_translog__translog_rtp TO fk_payment__payment_rtp;
-- Rename sequences
ALTER SEQUENCE IF EXISTS control.translog_tl_record_id_seq RENAME TO payment_id_seq;
ALTER SEQUENCE IF EXISTS control.translog_rtp_id_seq RENAME TO payment_rtp_id_seq;
ALTER SEQUENCE IF EXISTS control.translog_bonus_trb_id_seq RENAME TO payment_bonus_trb_id_seq;
-- Payment time stamp is included in PK. No extra index needed.
DROP INDEX control.idx_translog__pay_date_time;


COMMENT ON COLUMN control.payment.psn IS 'car License Plate Number';
COMMENT ON COLUMN control.payment.psn IS 'Parking Space Number';

-- Payment time stamp are stored with precision of seconds.
DO $$BEGIN raise notice 'Adjusting types ...'; END;$$;
ALTER TABLE control.payment ALTER COLUMN pay_ts TYPE timestamp(0) with time zone;

-- Drop foreign keys referencing primary key to be able to drop it.
-- They will be re-created later.
DO $$BEGIN raise notice 'Altering constraints ...'; END;$$;
ALTER TABLE control.translog_ext DROP CONSTRAINT "fk_tarnslog_ext__translog-base";
ALTER TABLE control.translog_ext DROP CONSTRAINT "fk_tarnslog_ext__translog-ext";
ALTER TABLE control.translog_bonus DROP CONSTRAINT "fk_translog_bonus__payment";
ALTER TABLE control.bonus_disposable DROP CONSTRAINT "fk_bonus_disposable__payment";
-- And their indexes
DROP INDEX control."fki_tarnslog_ext__translog-base";
DROP INDEX control."fki_translog_bonus__translog";
DROP INDEX control."fki_bonus_disposable__payment";
-- Create new primary key
DO $$BEGIN raise notice 'Altering primary key ...'; END;$$;
ALTER TABLE control.payment DROP CONSTRAINT pk_translog;
ALTER TABLE control.payment ADD CONSTRAINT pk_payment PRIMARY KEY (pay_ts, pdm_id);


-- ===================
-- == Payment order ==
-- ===================
-- Keep track about payments in order of receiving for post-processing. 
CREATE SEQUENCE control.payment_recent_id_seq;
-- Copy value of sequence from translog to not break export API state for clients
SELECT setval('control.payment_recent_id_seq', (SELECT last_value FROM control.payment_id_seq), false);

CREATE TABLE IF NOT EXISTS control.payment_recent
(
    id bigint NOT NULL DEFAULT nextval('control.payment_recent_id_seq'::regclass),
    pay_ts timestamp with time zone NOT NULL,
    pdm_id integer NOT NULL,
    server_ts timestamp with time zone NOT NULL,
    CONSTRAINT payment_recent_pkey PRIMARY KEY (id),
    CONSTRAINT fk_payment_recent__payment FOREIGN KEY (pay_ts, pdm_id)
        REFERENCES control.payment (pay_ts, pdm_id) MATCH SIMPLE
        ON UPDATE CASCADE
        ON DELETE CASCADE
);
COMMENT ON TABLE control.payment_recent IS 'Keeps track of incoming payments in order of reception for last X days. It is used for export API, post processing, and real time operations which operate only on recent payments.';

-- Add indexes
CREATE UNIQUE INDEX IF NOT EXISTS fki_payment_recent__payment ON control.payment_recent USING btree (pay_ts ASC NULLS LAST, pdm_id ASC NULLS LAST) WITH (deduplicate_items=False);
CREATE INDEX IF NOT EXISTS idx_payment_queu_server_ts ON control.payment_recent USING btree (server_ts ASC NULLS LAST) WITH (deduplicate_items=True);

ALTER SEQUENCE IF EXISTS control.payment_recent_id_seq OWNED BY control.payment_recent.id;

DO $$BEGIN raise notice 'Populating recent payments ...'; END;$$;
-- Copy last 30 days of payment to this table
INSERT into control.payment_recent (id, pay_ts, pdm_id, server_ts)
(SELECT pay_id, pay_ts, pdm_id, server_ts FROM control.payment WHERE pay_ts > now() - INTERVAL '30 days' ORDER BY pay_id);

-- ====================================
-- == Parking ticket extension table ==
-- ====================================
DO $$BEGIN raise notice 'Altering ticket extension ...'; END;$$;
ALTER TABLE control.translog_ext RENAME TO payment_ext;

-- Delete null entries which should not be there.
DELETE FROM control.payment_ext WHERE permit_base is null or permit_ext is null;

-- Add new columns
ALTER TABLE control.payment_ext ADD COLUMN base_pdm_id integer;
ALTER TABLE control.payment_ext ADD COLUMN base_pay_ts timestamp(0) with time zone;
ALTER TABLE control.payment_ext ADD COLUMN ext_pdm_id integer;
ALTER TABLE control.payment_ext ADD COLUMN ext_pay_ts timestamp(0) with time zone;

-- Maintain connection to base permit by PK.
UPDATE control.payment_ext
SET base_pdm_id = subq.pdm_id, base_pay_ts = subq.pay_ts
FROM (
	SELECT permit_base, pdm_id, pay_ts
	FROM control.payment_ext
	JOIN control.payment ON pay_id = payment_ext.permit_base
) AS subq
WHERE payment_ext.permit_base = subq.permit_base;
-- FK and index
ALTER TABLE control.payment_ext
    ADD CONSTRAINT "fk_payment_ext__payment-base" FOREIGN KEY (base_pdm_id, base_pay_ts)
    REFERENCES control.payment (pdm_id, pay_ts) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE;
CREATE INDEX "fki_payment_ext__payment-base" ON control.payment_ext(base_pay_ts, base_pdm_id);

ALTER TABLE control.payment_ext DROP COLUMN IF EXISTS permit_base;

-- Maintain connection to extension permit by PK.
UPDATE control.payment_ext
SET ext_pdm_id = subq.pdm_id, ext_pay_ts = subq.pay_ts
FROM (
	SELECT permit_ext, pdm_id, pay_ts
	FROM control.payment_ext
	JOIN control.payment ON pay_id = payment_ext.permit_ext
) AS subq
WHERE payment_ext.permit_ext = subq.permit_ext;
-- FK and index 
ALTER TABLE control.payment_ext
    ADD CONSTRAINT "fk_payment_ext__payment-ext" FOREIGN KEY (ext_pdm_id, ext_pay_ts)
    REFERENCES control.payment (pdm_id, pay_ts) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE;
CREATE INDEX "fki_payment_ext__payment-ext" ON control.payment_ext(ext_pay_ts, ext_pdm_id);
-- Add foreign key and index for base permit

ALTER TABLE control.payment_ext DROP COLUMN IF EXISTS permit_ext;

-- Add new primary key. It will automatically set non null constraint for all included columns.
ALTER TABLE IF EXISTS control.payment_ext ADD CONSTRAINT pk_payment_ext PRIMARY KEY (base_pdm_id, base_pay_ts, ext_pdm_id, ext_pay_ts);

-- =============================
-- == Real time parking table ==
-- =============================
DO $$BEGIN raise notice 'Altering real time parking ...'; END;$$;
-- Rename real time parking sessions table
ALTER TABLE control.translog_rtp RENAME TO payment_rtp;
ALTER TABLE control.payment_rtp  RENAME CONSTRAINT uc_translog_rtp__session_name TO uc_payment_rtp__session_name;

-- =================
-- == Bonus table ==
-- =================
DO $$BEGIN raise notice 'Altering bonus ...'; END;$$;
ALTER TABLE control.translog_bonus RENAME TO payment_bonus;
ALTER TABLE control.payment_bonus ADD COLUMN pay_ts timestamp(0) with time zone;
ALTER TABLE control.payment_bonus ADD COLUMN pay_pdm_id integer;
ALTER TABLE control.payment_bonus RENAME CONSTRAINT pk_translog_bonus TO pk_payment_bonus;

ALTER TABLE control.payment_bonus ADD CONSTRAINT fk_payment_bonus__payment FOREIGN KEY (pay_ts, pay_pdm_id)
	REFERENCES control.payment (pay_ts, pdm_id) MATCH SIMPLE ON UPDATE CASCADE ON DELETE CASCADE;
CREATE INDEX fki_payment_bonus__payment ON control.payment_bonus(pay_ts, pay_pdm_id);

-- Update connection to payments
UPDATE control.payment_bonus
SET pay_ts = subq.src_pay_ts, pay_pdm_id = subq.src_pdm_id
FROM (
	SELECT pb2.payment as payment_id, payment.pay_ts as src_pay_ts, payment.pdm_id as src_pdm_id
	FROM control.payment_bonus AS pb2
	JOIN control.payment ON payment.pay_id = pb2.payment
) AS subq
WHERE subq.payment_id = payment_bonus.payment;

-- Delete entries without reference
DELETE FROM control.payment_bonus WHERE pay_ts IS NULL OR pay_pdm_id IS NULL;

-- Set non null constraints
ALTER TABLE control.payment_bonus ALTER COLUMN pay_ts SET NOT NULL;
ALTER TABLE control.payment_bonus ALTER COLUMN pay_pdm_id SET NOT NULL;

-- =======================
-- == Bonus disposable  ==
-- =======================
DO $$BEGIN raise notice 'Altering disposable bonus ...'; END;$$;
-- Add new columns for FK.
ALTER TABLE IF EXISTS control.bonus_disposable ADD COLUMN pay_ts timestamp(0) with time zone;
ALTER TABLE IF EXISTS control.bonus_disposable ADD COLUMN pay_pdm_id integer;
-- Update connection
UPDATE control.bonus_disposable
SET pay_ts = subq.pay_ts, pay_pdm_id = subq.pdm_id
FROM (
	SELECT payment_id, payment.pay_ts, payment.pdm_id
	FROM control.bonus_disposable 
	JOIN control.payment ON payment.pay_id = bonus_disposable.payment_id
) AS subq
WHERE subq.payment_id = bonus_disposable.payment_id;
-- Create FK
ALTER TABLE IF EXISTS control.bonus_disposable
    ADD CONSTRAINT fk_bonus_disposable__payment FOREIGN KEY (pay_ts, pay_pdm_id)
    REFERENCES control.payment (pay_ts, pdm_id) MATCH SIMPLE
    ON UPDATE CASCADE ON DELETE CASCADE;
-- Create index
CREATE INDEX IF NOT EXISTS fki_bonus_disposable__payment ON control.bonus_disposable(pay_ts, pay_pdm_id);
-- Drop old FK column
ALTER TABLE control.bonus_disposable DROP COLUMN payment_id;

-- ===================
-- == Payment table ==
-- ===================
ALTER TABLE control.payment DROP COLUMN pay_id;

-- =========================================
-- == Adjust function extendible permits  ==
-- =========================================
DO $$BEGIN raise notice 'Altering functions ...'; END;$$;
DROP FUNCTION IF EXISTS control.extendible_permits(integer, character varying, integer);

CREATE OR REPLACE FUNCTION control.extendible_permits(
	parAreaNr integer,
	parLpn character varying,
	parPsn integer)
	RETURNS TABLE(
		res_pay_date_time timestamp without time zone, 
		res_pdm_id integer,
		res_pdm_number integer, 
		res_pdm_name character varying,
		res_base_price numeric, 
		res_exp_date_time timestamp without time zone, 
		res_tariff_name character varying, 
		res_base_permit_nr integer,  
		res_tolerance_time integer, 
		res_penalty_value numeric, 
		res_parts integer, 
		res_ext_price numeric, 
		res_max_permits integer, 
		res_max_duration integer, 
		res_max_price numeric
	)
	LANGUAGE 'plpgsql'
	VOLATILE PARALLEL UNSAFE
AS $BODY$
DECLARE
	aTimeZone VARCHAR;
	historyLimit TIMESTAMP WITH TIME ZONE;
BEGIN
	SELECT ac_time_zone INTO aTimeZone FROM control.area WHERE ac_id = parAreaNr;
	EXECUTE 'SET LOCAL TIME ZONE ''' || aTimeZone || ''';' ;
	SELECT NOW() - INTERVAL '5 days' INTO historyLimit;

	RETURN QUERY
	WITH extension_candidates AS(
		SELECT
		pay_ts::timestamp AS pay_time,
		payment.pdm_id as pdm_id,
		pdm_number AS seller_number,
		pdm_name AS seller_name,
		amount AS base_price,
		expiration_ts::timestamp AS exp_time, 
		tai_name AS tariff_name,
		ticket_nr AS base_permit_nr,
		pex_tolerance_time AS tolerance_time, pex_penalty_value AS penalty_value,
		-- Count number of parts of already sold extensions.
		(SELECT COUNT (*) 
			FROM control.payment AS payment2 
			LEFT JOIN control.payment_ext ON payment_ext.ext_pay_ts = payment2.pay_ts AND payment_ext.ext_pdm_id = payment2.pdm_id
			WHERE payment_ext.base_pay_ts = payment2.pay_ts AND payment_ext.base_pdm_id = payment2.pdm_id AND pay_reason = 'PERMIT_EXTENSION'::control.payment_reason AND pay_ts > historyLimit 
		)::integer AS ext_parts,
		-- Count total amount spent on this ticket. Currency is not considered. Assuming that one town use only one currency.
		(SELECT COALESCE (SUM (amount), 0) 
			FROM control.payment AS payment2 
			LEFT JOIN control.payment_ext ON payment_ext.ext_pay_ts = payment2.pay_ts AND payment_ext.ext_pdm_id = payment2.pdm_id
			WHERE payment_ext.base_pay_ts = payment2.pay_ts AND payment_ext.base_pdm_id = payment2.pdm_id AND pay_reason = 'PERMIT_EXTENSION'::control.payment_reason AND pay_ts > historyLimit
		) AS ext_price,
		pex_max_permits AS max_permits,
		pex_max_duration AS max_duration,
		pex_max_price AS max_price
		FROM control.payment
			JOIN control.pdm on (payment.pdm_id = pdm.pdm_id)
			LEFT JOIN control.zone on (zon_id = pdm_zone_id)
			LEFT JOIN control.zone__perm_ext on pex_id = zon_id
			LEFT JOIN control.tariff_info on tai_id = tariff_info_id
		WHERE
			pex_enabled
			AND pay_ts > historyLimit
			AND pay_reason = 'PURCHASE'::control.payment_reason 
			AND zon_area_id = parAreaNr
			-- Choose LPN or PSN filleter based on what is NON NULL
			AND (
				((parLpn IS NOT NULL) AND (payment.lpn = parLpn))
				OR
				((parPsn IS NOT NULL) AND (payment.psn = parPsn))
			)
			AND (
				expiration_ts > NOW()
				OR
				expiration_ts > NOW() - CAST(GREATEST(pex_penalty_time, pex_tolerance_time) || 'minutes' AS interval)
			)
		ORDER BY expiration_ts DESC
	)
	SELECT DISTINCT ON (pay_time, pdm_id) * FROM extension_candidates
	WHERE
		ext_parts < max_permits
		AND
		(base_price + ext_price) < max_price
		AND
		(EXTRACT(epoch FROM (exp_time - pay_time)) / 60) < max_duration
	LIMIT 10;
END; 
$BODY$;

